from cobs_usb_serial import CobsUsbSerial 
import struct 
import numpy as np 
import pandas as pd 
import matplotlib.pyplot as plt 

ser = CobsUsbSerial("COM23") 

stamp_count = 1000
pck_len = 250

stamps = np.zeros(stamp_count)

for i in range(stamp_count):
  bts = ser.read()
  if len(bts) == pck_len:
    stamp = struct.unpack('=I', bts[:4])
    stamps[i] = stamp[0]

print("stamps, ", stamps)

df = pd.DataFrame({'timestamps': stamps})

df['deltas'] = df['timestamps'].diff() 

# clean NaN's 
df = df.dropna()

# wipe obviously-wrong deltas (i.e. the 1st, which goes 0-start-us) 
df = df[df['deltas'] < 100000]

# Plotting
fig, ax1 = plt.subplots(figsize=(11, 5))

# Primary x-axis (time deltas)
df['deltas'].plot(kind='hist', bins=100, ax=ax1)
ax1.set_xlabel('Time-Stamp Deltas (us)')
ax1.set_ylabel(f'Frequency (of {stamp_count})')

# Secondary x-axis (bandwidth)
ax2 = ax1.twiny()
ax2.set_xlabel('Estimated Bandwidth (Mbits/s)')

# Set the limits of the secondary axis based on the primary axis
# new_tick_locations = np.linspace(df['deltas'].min(), df['deltas'].max(), num=len(ax1.get_xticks()))

# Convert tick locations to bandwidth
# bandwidths = [(pck_len * 8) * (1e6 / x) for x in new_tick_locations]

x_ticks = ax1.get_xticks()
bandwidth_ticks = [((pck_len * 8) * (1e6 / x)) / 1e6 for x in x_ticks]

ax2.set_xlim(max(bandwidth_ticks), min(bandwidth_ticks))

plt.title(f'Single-Source COBS Data Sink Deltas, pck_len={pck_len}')

plt.tight_layout()

plt.show()
